#pragma once
#include <string>
#include <iomanip>
#include "GVector.h"
#include "GException.h"


//========================================================
//                      declaration of GMatrix
//========================================================
/*! \class GMatrix
* \brief Definition, construction, modification and operations GMatrix
*/

class GMatrix
{

private:

    vector<vector<double>> _dataMatrix; /*!< The data of matrix*/
	int _nbLines; /*!<the number of line in the matrix*/
    int _nbColumns;	/*!<the number of columns in the matrix*/

public:

	//inline methods---------------------------------------
	/*!
	* \brief Allocate memory for the matrix
	*/
	inline void allocGMatrix();
	/*!
	* \brief Deallocate memory for the matrix
	*/
	inline void desallocGMatrix();
	/*!
	* \brief get the content of this matrix
	* \param mat: the matrix tha will be contain the content of this matrix
	*/
	inline void getMatrix(const GMatrix& mat);
	/*!
	* \brief Get the element in the column number column Index end in the line number lineIndex
	* \param lineIndex    : the line number of this element
	* \param  columnIndex : the column number of this element
	* \return The value of the element
	*/

	inline const double& getElement (  int lineIndex,   int columnIndex) const{
		return _dataMatrix[(lineIndex-1)][(columnIndex-1)];
	}
	/*!
	* \brief Get the element in the column number column Index end in the line number lineIndex
	* \param lineIndex    : the line number of this element
	* \param  columnIndex : the column number of this element
	* \return The value of the element
	*/
	inline double& getElement (  int lineIndex,   int columnIndex){
		return _dataMatrix[(lineIndex-1)][(columnIndex-1)];
	}
	inline void setElement ( const int& lineIndex, const int& columnIndex, const double& value);

	//constructors destructor------------------------------
	/*!  
	* \brief Default Constructor
	* Instantiates an instance of GMatrix 
	*/
	GMatrix();
	/*!  
	* \brief Constructor
	* Instantiates an instance of GMatrix with the number of lines and the number of columns
	* \param nbLines	: The number of lines in the matrix
	* \param nbColumns	: The number of Columns in the matrix
	*/
	GMatrix(  int nbLines,   int nbColumns);
	/*!  
	* \brief Constructor
	* * Instantiates an instance of GMatrix with the contents of another GMatrix 
	* \param mat : The matrix copy in the constructor
	*/
	GMatrix(const GMatrix& mat);
	/*!
    *  \brief Destructor
    *
    *  Destroyer Class GMatrix
    */
	virtual ~GMatrix();

	//other methods----------------------------------------
	/*!
	* \brief Get the number of lines in this matrix
	* \return the number of lines in this matrix
	*/
	int getNbLines() const;
	/*!
	* \brief Get the number of columns in this matrix
	* \return The number of columns
	*/
	int getNbColumns() const;
	/*!
	* \brief Initialize the matrix with the value v
	* \param v:The initialization value 
	*/
	void initWithValue(double v);
	/*! 
	* \brief get the line number indexLine
	* \param indexLine: the index of the line in the matrix
	* \return The line number indexLine in the matrix
	*/
	GVector getLine(  int indexLine) const;
	/*! 
	* \brief Set the line line in the indexLine
	* \param indexLine: the index of the line in the matrix
	* \param line: the elements of the line
	*/
	void setLine(  int indexLine, GVector line);
	/*! 
	* \brief Get the norme of the line
	* \param indexLine: the index of the line in the matrix
	* \return The norme of the line
	*/
	double getNormOfLine(int indexLine);
	/*! 
	* \brief Get the column number indexColumn
	* \param indexColumn: the index of the indexColumn
	* \return A vector that contain the element of the column
	*/
	GVector getColumn(  int indexColumn) const;
	/*! 
	* \brief Set the column column in the indexColumn
	* \param indexColumn	: the index of the column in the matrix
	* \param column			: Elements of the column
	*/
	void setColumn(  int indexColumn, GVector column);
	/*! 
	* \brief Get the norme of the column
	* \param indexLine: the index of the column in the matrix
	* \return The norme of the column 
	*/
	double getNormOfColumn(int indexColumn);
	/*! 
	* \brief Set the 4x4 values (this method is designed for transformations and works only if size = 4x4)
	* \param A11 to A44		: the 16 values of the matrix elements
	* \return 0 if ok, nonzero otherwise 
	*/
	int set4x4values(double A11, double A12, double A13, double A14,
					 double A21, double A22, double A23, double A24,
					 double A31, double A32, double A33, double A34,
					 double A41, double A42, double A43, double A44);

	//operators--------------------------------------------
	/*!
	* \brief Assignement operator
	* \param mat : a matrix
	* \return a matrix equal to the first matrix (mat)
	*/
	GMatrix& operator = (const GMatrix& mat);
	/*!
	* \brief Get the element in the line indexLine and in the column indexColumn 
	* \param indexLine		:The line of the element
	* \param indexColumn	: a column in the matrix
	* \return The element in the line indexLine and in the column indexColumn 
	*/	
	const double& operator() (  int indexLine,   int indexColumn) const ;
	/*!
	* \brief Get the element in the line indexLine and in the column indexColumn 
	* \param indexLine		:The line of the element
	* \param indexColumn	: a column in the matrix
	* \return The element in the line indexLine and in the column indexColumn 
	*/
	double& operator() (  int indexLine,   int indexColumn);
	/*!
	* \brief Add mat to this matrix
	* \param mat		:The matrix To add
	* \return The modified matrix
	*/
	GMatrix& operator += (const GMatrix& mat);
	/*!
	* \brief Subtract mat to this matrix
	* \param mat		:The matrix To substract
	* \return The modified matrix
	*/
	GMatrix& operator -= (const GMatrix& mat);
	/*!
	* \brief Multiply this matrix by mat
	* \param mat		:The matrix To multiply
	* \return The modified matrix
	*/
	GMatrix& operator *= (const GMatrix& mat); 
	/*!
	* \brief Multiply this matrix by mat
	* \param mat		:The matrix To multiply
	* \return The modified matrix
	*/
	GMatrix& operator *= (const double& v);
	GMatrix operator - ();
	GMatrix operator + ();
	/*!
	* \brief Add mat1 to mat2
	* \param mat1	:The first matrix
	* \param mat2	:The second matrix
	* \return The result matrix
	*/
	friend GMatrix operator + (const GMatrix& mat1, const GMatrix& mat2);
	/*!
	* \brief Subtract mat2 to mat1
	* \param mat1	:The first matrix
	* \param mat2	:The second matrix
	* \return The result matrix
	*/
	friend GMatrix operator - (const GMatrix& mat1, const GMatrix& mat2);
	/*!
	* \brief Multiply mat1 by mat2
	* \param mat1	:The first matrix
	* \param mat2	:The second matrix
	* \return The result matrix
	*/friend GMatrix operator * (const GMatrix& mat1, const GMatrix& mat2);
	/*!
	* \brief Multiply mat1 by the double v
	* \param mat:The matrix To multiply
	* \param v	:The double v
	* \return The result matrix
	*/friend GMatrix operator * (const double& v, const GMatrix& mat);
	/*!
	* \brief Multiply mat by v
	* \param mat	:The matrix to multiply
	* \param v		:The double v
	* \return The result matrix
	*/friend GMatrix operator * (const GMatrix& mat, const double& v);
	/*!
	* \brief divide mat by v
	* \param mat	:The matrix to divide
	* \param v		:The double v
	* \return The result matrix
	*/
	friend GMatrix operator / (const GMatrix& mat, const double& v);
	/*!
	* \brief Multiply mat by vect
	* \param mat	:The matrix to multiply
	* \param vect	:The vector
	* \return The result matrix
	*/
	friend GMatrix operator * (const GVector& vect, const GMatrix& mat);
	/*!
	* \brief Multiply vect by mat
	* \param mat	:The matrix 
	* \param vect	:The vector tomultiply
	* \return The result vector
	*/
	friend GVector operator * (const GMatrix& mat, const GVector& vect);
	friend GMatrix operator ~ (const GMatrix& mat);
	friend ostream& operator << (ostream& out, const GMatrix& mat);
};



			













